<?php

declare(strict_types=1);

/**
 * @copyright Copyright (c) 2016 Bjoern Schiessle <bjoern@schiessle.org>
 * @copyright Copyright (c) 2019 Joas Schilling <coding@schilljs.com>
 *
 * @author Arthur Schiwon <blizzz@arthur-schiwon.de>
 * @author Bjoern Schiessle <bjoern@schiessle.org>
 * @author Joas Schilling <coding@schilljs.com>
 * @author Lukas Reschke <lukas@statuscode.ch>
 * @author Morris Jobke <hey@morrisjobke.de>
 * @author Roeland Jago Douma <roeland@famdouma.nl>
 * @author Vincent Petry <vincent@nextcloud.com>
 *
 * @license GNU AGPL version 3 or any later version
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 */
namespace OCA\LookupServerConnector\BackgroundJobs;

use OC\Security\IdentityProof\Signer;
use OCP\Accounts\IAccountManager;
use OCP\AppFramework\Utility\ITimeFactory;
use OCP\BackgroundJob\IJobList;
use OCP\BackgroundJob\Job;
use OCP\Http\Client\IClientService;
use OCP\IConfig;
use OCP\IUser;
use OCP\IUserManager;

class RetryJob extends Job {
	private IClientService $clientService;
	private string $lookupServer;
	private IConfig $config;
	private IUserManager $userManager;
	private IAccountManager $accountManager;
	private Signer $signer;
	protected int $retries = 0;
	protected bool $retainJob = false;

	/**
	 * @param ITimeFactory $time
	 * @param IClientService $clientService
	 * @param IConfig $config
	 * @param IUserManager $userManager
	 * @param IAccountManager $accountManager
	 * @param Signer $signer
	 */
	public function __construct(ITimeFactory $time,
		IClientService $clientService,
		IConfig $config,
		IUserManager $userManager,
		IAccountManager $accountManager,
		Signer $signer) {
		parent::__construct($time);
		$this->clientService = $clientService;
		$this->config = $config;
		$this->userManager = $userManager;
		$this->accountManager = $accountManager;
		$this->signer = $signer;

		$this->lookupServer = $config->getSystemValue('lookup_server', 'https://lookup.nextcloud.com');
		if (!empty($this->lookupServer)) {
			$this->lookupServer = rtrim($this->lookupServer, '/');
			$this->lookupServer .= '/users';
		}
	}

	/**
	 * Run the job, then remove it from the jobList
	 */
	public function start(IJobList $jobList): void {
		if (!isset($this->argument['userId'])) {
			// Old background job without user id, just drop it.
			$jobList->remove($this, $this->argument);
			return;
		}

		$this->retries = (int) $this->config->getUserValue($this->argument['userId'], 'lookup_server_connector', 'update_retries', '0');

		if ($this->shouldRemoveBackgroundJob()) {
			$jobList->remove($this, $this->argument);
			return;
		}

		if ($this->shouldRun()) {
			parent::start($jobList);
			if (!$this->retainJob) {
				$jobList->remove($this, $this->argument);
			}
		}
	}

	/**
	 * Check if we should kill the background job:
	 *
	 * - internet connection is disabled
	 * - no valid lookup server URL given
	 * - lookup server was disabled by the admin
	 * - max retries are reached (set to 5)
	 */
	protected function shouldRemoveBackgroundJob(): bool {
		// TODO: Remove global scale condition once lookup server is used for non-global scale federation
		// return $this->config->getAppValue('files_sharing', 'lookupServerUploadEnabled', 'no') !== 'yes'
		return !$this->config->getSystemValueBool('gs.enabled', false)
			|| $this->config->getSystemValueBool('has_internet_connection', true) === false
			|| $this->config->getSystemValueString('lookup_server', 'https://lookup.nextcloud.com') === ''
			|| $this->retries >= 5;
	}

	protected function shouldRun(): bool {
		$delay = 100 * 6 ** $this->retries;
		return ($this->time->getTime() - $this->lastRun) > $delay;
	}

	protected function run($argument): void {
		$user = $this->userManager->get($this->argument['userId']);
		if (!$user instanceof IUser) {
			// User does not exist anymore
			return;
		}

		$data = $this->getUserAccountData($user);
		$signedData = $this->signer->sign('lookupserver', $data, $user);
		$client = $this->clientService->newClient();

		try {
			if (count($data) === 1) {
				$dataOnLookupServer = $this->config->getUserValue($user->getUID(), 'lookup_server_connector', 'dataSend', '0') === '1';

				if (!$dataOnLookupServer) {
					// We never send data to the lookupserver so no need to delete it
					return;
				}

				// There is data on the lookup server so we must delete it
				$client->delete($this->lookupServer,
					[
						'body' => json_encode($signedData),
						'timeout' => 10,
						'connect_timeout' => 3,
					]
				);

				$this->config->setUserValue($user->getUID(), 'lookup_server_connector', 'dataSend', '0');
			} else {
				$client->post($this->lookupServer,
					[
						'body' => json_encode($signedData),
						'timeout' => 10,
						'connect_timeout' => 3,
					]
				);
				$this->config->setUserValue($user->getUID(), 'lookup_server_connector', 'dataSend', '1');
			}

			// Reset retry counter
			$this->config->deleteUserValue(
				$user->getUID(),
				'lookup_server_connector',
				'update_retries'
			);
		} catch (\Exception $e) {
			// An error occurred, retry later
			$this->retainJob = true;
			$this->config->setUserValue(
				$user->getUID(),
				'lookup_server_connector',
				'update_retries',
				(string)($this->retries + 1),
			);
		}
	}

	protected function getUserAccountData(IUser $user): array {
		$account = $this->accountManager->getAccount($user);

		$publicData = [];
		foreach ($account->getAllProperties() as $property) {
			if ($property->getScope() === IAccountManager::SCOPE_PUBLISHED) {
				$publicData[$property->getName()] = [
					'value' => $property->getValue(),
					'verified' => $property->getVerified(),
					'signature' => $property->getVerificationData(),
				];
			}
		}

		$data = ['federationId' => $user->getCloudId()];
		if (!empty($publicData)) {
			$data['name'] = $publicData[IAccountManager::PROPERTY_DISPLAYNAME]['value'] ?? '';
			$data['email'] = $publicData[IAccountManager::PROPERTY_EMAIL]['value'] ?? '';
			$data['address'] = $publicData[IAccountManager::PROPERTY_ADDRESS]['value'] ?? '';
			$data['website'] = $publicData[IAccountManager::PROPERTY_WEBSITE]['value'] ?? '';
			$data['twitter'] = $publicData[IAccountManager::PROPERTY_TWITTER]['value'] ?? '';
			$data['phone'] = $publicData[IAccountManager::PROPERTY_PHONE]['value'] ?? '';
			$data['twitter_signature'] = $publicData[IAccountManager::PROPERTY_TWITTER]['signature'] ?? '';
			$data['website_signature'] = $publicData[IAccountManager::PROPERTY_WEBSITE]['signature'] ?? '';
			$data['verificationStatus'] = [
				IAccountManager::PROPERTY_WEBSITE => $publicData[IAccountManager::PROPERTY_WEBSITE]['verified'] ?? '',
				IAccountManager::PROPERTY_TWITTER => $publicData[IAccountManager::PROPERTY_TWITTER]['verified'] ?? '',
			];
		}

		return $data;
	}
}
